function [xss,yss,sss,Pss,H,G,H2,G2,retcode] = endogenous_ms_solver_main(GLOBALS,THETA)
% 
% solves a general MS model to the second order
% 
% 
% Andrew Foerster
% edited 2014/04/03
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % 
% 
% INPUTS
%  GLOBALS  = structure with options
%  THETA    = (nTHETA)-vector of parameters
% 
% 
% OUTPUTS
%  xss  = (nx x 1) vector of steady state of x, the predetermined variables
%  yss  = (ny x 1) vector of steady state of y, the nonpredetermined variables
%  sss  = (ns x 1) vector of erdogic distribution across regimes
%  Pss  = (ns x ns) matrix of the transition matrix at steady state
%  H    = (ns x 1) cell with cells H{1} H{2} ... H{ns}
%           where H{s} is the first-order solution for the predetermined
%           variables when the regime is s
%  G    = (ns x 1) cell with cells G{1} G{2} ... G{ns}
%           where G{s} is the first-order solution for the predetermined
%           variables when the regime is s
%  H2   = (nx+ne+1 x ns*(nx+ne+1)*nx)-matrix with blocks
%           H2 = [H2_{1} H2_{2} ... H2_{ns}] which has sub-blocks of the form
%           H2_{i} =  [H2_{i,1} H2_{i,2} ... H2_{i,nx}] where H2_{i,j} is the 
%           Hessian for the j-th variable of x with respect to (xlag,eps,chi) 
%           given the regime is i
%  G2   = (nx+ne+1 x ns*(nx+ne+1)*ny)-matrix with blocks
%           G2 = [G2_{1} G2_{2} ... G2_{ns}] which has sub-blocks of the form
%           G2_{i} = [G2_{i,1} G2_{i,2} ... G2_{i,nx}] where G2_{i,j} is the
%           Hessian for the j-th variable of x with respect to (xlag,eps,chi)
%           given the regime is i
%  retcode  = boolean taking values
%       0: solution not found, no steady state, or
%               iterative procedure does not converge
%       1: solution found, iterative procedure converges  
% 
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % 

 
% -- Inputs -- %
ny = GLOBALS.ny;
nx = GLOBALS.nx;
ne = GLOBALS.ne;
ns = GLOBALS.ns;
printtime = GLOBALS.FRWZprinttime;


% -- Setup -- %
modelsoln = tic;

% nt1 = nvec(4);
% nt2 = nvec(5);
n = nx + ny;
% nt = nt1 + nt2;

% -- Initialization -- %
H = cell(ns,1);
G = cell(ns,1);
H2 = cell(ns,1);
G2 = cell(ns,1);

for ii = 1:ns
    H{ii} = NaN(nx,nx+ne+1);
    G{ii} = NaN(ny,nx+ne+1);
    H2{ii} = NaN(nx,(nx+ne+1)^2);
    G2{ii} = NaN(ny,(nx+ne+1)^2);
end 

% -- Steady State -- %
[xss,yss,sss,Pss,retcode] = solution_getsteadystate(GLOBALS,THETA);
if retcode == 0
    warning('Solver Warning -- Steady State Not Found');
    return; 
end

% -- First Order -- %
% Get Jacobian
[A,B] = solution_getmatrices_x(GLOBALS,THETA,xss,yss,sss,Pss); 


% Run Iterative Procedure
[gx,hx,retcode] = FRWZ_iterativeprocedure(GLOBALS,THETA,xss,yss,sss,Pss,A,B);
if retcode == 0 
    warning('Solver Warning -- Iterative Procedure Does Not Converge');
    return; 
end

% Check Mean Square Stability
retcode = FRWZ_msscheck(Pss,hx);
if retcode == 0
    warning('Solver Warning -- Not a MSS Solution');
    return
end
for ii = 1:ns
    H{ii}(:,1:nx) = hx{ii};
    G{ii}(:,1:nx) = gx{ii};
end

% First order for epsilon
[Ce,De] = solution_getmatrices_e(GLOBALS,THETA,xss,yss,sss,G);
Cefull = zeros(ns*n,ns*n);
Defull = zeros(ns*n,ne);
for ii = 1:ns
    iiblock = (ii-1)*n+1:ii*n;
    for jj = 1:ns
        Cefull(iiblock,iiblock) = Cefull(iiblock,iiblock) + Pss(ii,jj)*Ce{jj,ii};
        Defull(iiblock,1:ne) = Defull(iiblock,1:ne) + Pss(ii,jj)*De{jj,ii};
    end
end
esoln = -Cefull\Defull; 
for ii = 1:ns
    H{ii}(:,nx+1:nx+ne) = esoln((ii-1)*n+1:(ii-1)*n+nx,1:ne);
    G{ii}(:,nx+1:nx+ne) = esoln((ii-1)*n+nx+1:ii*n,1:ne);
end


% First order for Chi
[Cc,Dc] = solution_getmatrices_c(GLOBALS,THETA,xss,yss,sss,G);
Ccfull = zeros(ns*n,ns*n);
Dcfull = zeros(ns*n,1);
for ii = 1:ns
    iiblock = (ii-1)*n+1:ii*n;
    for jj = 1:ns
        jjblock = (jj-1)*n+nx+1:jj*n;
        Ccfull(iiblock,iiblock) = Ccfull(iiblock,iiblock) + Pss(ii,jj)*Cc{jj,ii}(1:n,1:n);
        Ccfull(iiblock,jjblock) = Ccfull(iiblock,jjblock) + Pss(ii,jj)*Cc{jj,ii}(1:n,n+1:n+ny);
        Dcfull(iiblock,1) = Dcfull(iiblock,1) + Pss(ii,jj)*Dc{jj,ii}(1:n,1);
    end
end
csoln = -Ccfull\Dcfull;
for ii = 1:ns
    H{ii}(:,nx+ne+1) = csoln((ii-1)*n+1:(ii-1)*n+nx);
    G{ii}(:,nx+ne+1) = csoln((ii-1)*n+nx+1:ii*n);
end
% if (retcode == 0) || (isreal(H{1}) == 0) 
%     retcode = 0;
%     warning('Solver Warning -- Imaginary Part of Chi Derivative');
%     return;
% end
    
% -- Second Order -- %
% Second order for x,x
[Mxx,Nxx] = solution_getmatrices_xx(GLOBALS,THETA,xss,yss,sss,Pss,H,G);
Mxxfull = zeros(ns*(n*nx*nx),ns*(n*nx*nx));
Nxxfull = zeros(ns*(n*nx*nx),1);
for ii = 1:ns
    iiblock = (ii-1)*n*nx*nx+1:ii*n*nx*nx;
    for jj = 1:ns
        jjblock = (jj-1)*n*nx*nx+nx*nx*nx+1:jj*n*nx*nx;
        Mxxfull(iiblock,iiblock) = Mxxfull(iiblock,iiblock) + ...
            Mxx{jj,ii}(:,1:n*nx*nx);
        Mxxfull(iiblock,jjblock) = Mxxfull(iiblock,jjblock) + ...
            Mxx{jj,ii}(:,n*nx*nx+1:(n+ny)*nx*nx);
        Nxxfull(iiblock) = Nxxfull(iiblock) + Nxx{jj,ii};
    end
end
warning('off','MATLAB:nearlySingularMatrix')
xxsoln = -Mxxfull\Nxxfull;
warning('on','MATLAB:nearlySingularMatrix')
for ii = 1:ns
    for jj = 1:nx
        H2{ii}(:,(jj-1)*(nx+ne+1)+1:(jj-1)*(nx+ne+1)+nx) = reshape(xxsoln((ii-1)*n*nx*nx+(jj-1)*nx^2+1:(ii-1)*n*nx*nx+jj*nx^2),nx,nx);
        G2{ii}(:,(jj-1)*(nx+ne+1)+1:(jj-1)*(nx+ne+1)+nx) = reshape(xxsoln((ii-1)*n*nx*nx+nx*nx*nx+(jj-1)*nx*ny+1:(ii-1)*n*nx*nx+nx*nx*nx+jj*nx*ny),ny,nx);
    end
end


% Second order for x,e
[Mxe,Nxe] = solution_getmatrices_xe(GLOBALS,THETA,xss,yss,sss,Pss,H,G,G2);
Mxefull = zeros(ns*n*nx*ne,ns*n*nx*ne);
Nxefull = zeros(ns*n*nx*ne,1);
for ii = 1:ns
	iiblock = (ii-1)*n*nx*ne+1:ii*n*nx*ne;
    for jj = 1:ns
        jjblock = (jj-1)*n*nx*ne+nx*nx*ne+1:jj*n*nx*ne;
        Mxefull(iiblock,iiblock) = Mxefull(iiblock,iiblock) + Mxe{jj,ii}(:,1:n*nx*ne);
        Mxefull(iiblock,jjblock) = Mxefull(iiblock,jjblock) + Mxe{jj,ii}(:,n*nx*ne+1:(n+ny)*nx*ne);
        Nxefull(iiblock) = Nxefull(iiblock) + Nxe{jj,ii};
    end
end
warning('off','MATLAB:nearlySingularMatrix')
xesoln = -Mxefull\Nxefull;
warning('on','MATLAB:nearlySingularMatrix')
for ii = 1:ns
    for jj = 1:nx
        H2{ii}(:,(jj-1)*(nx+ne+1)+nx+1:(jj-1)*(nx+ne+1)+nx+ne) = reshape(xesoln((ii-1)*n*nx*ne+(jj-1)*nx*ne+1:(ii-1)*n*nx*ne+jj*nx*ne),nx,ne);
        G2{ii}(:,(jj-1)*(nx+ne+1)+nx+1:(jj-1)*(nx+ne+1)+nx+ne) = reshape(xesoln((ii-1)*n*nx*ne+nx*nx*ne+(jj-1)*ny*ne+1:(ii-1)*n*nx*ne+nx*nx*ne+jj*ny*ne),ny,ne);
    end
end


% Second order for x,c
[Mxc,Nxc] = solution_getmatrices_xc(GLOBALS,THETA,xss,yss,sss,Pss,H,G,G2);
Mxcfull = zeros(ns*n*nx,ns*n*nx);
Nxcfull = zeros(ns*n*nx,1);
for ii = 1:ns
	iiblock = (ii-1)*n*nx+1:ii*n*nx;
	for jj = 1:ns
       jjblock = (jj-1)*n*nx+nx*nx+1:jj*n*nx;
       Mxcfull(iiblock,iiblock) = Mxcfull(iiblock,iiblock) + Mxc{jj,ii}(:,1:n*nx);
       Mxcfull(iiblock,jjblock) = Mxcfull(iiblock,jjblock) + Mxc{jj,ii}(:,n*nx+1:(n+ny)*nx);
       Nxcfull(iiblock) = Nxcfull(iiblock) + Nxc{jj,ii};
	end
end
warning('off','MATLAB:nearlySingularMatrix')
xcsoln = -Mxcfull\Nxcfull;
warning('on','MATLAB:nearlySingularMatrix')
for ii = 1:ns
    for jj = 1:nx
        H2{ii}(:,(jj-1)*(nx+ne+1)+nx+ne+1) = reshape(xcsoln((ii-1)*n*nx+(jj-1)*nx+1:(ii-1)*n*nx+jj*nx),nx,1);
        G2{ii}(:,(jj-1)*(nx+ne+1)+nx+ne+1) = reshape(xcsoln((ii-1)*n*nx+nx*nx+(jj-1)*ny+1:(ii-1)*n*nx+nx*nx+jj*ny),ny,1);
    end
end

% Second order for e,x (use Young's theorem)
for ii = 1:ns
    for jj = 1:ne
        for kk = 1:nx
            H2{ii}(:,(nx+jj-1)*(nx+ne+1)+kk) = xesoln((ii-1)*n*nx*ne+(kk-1)*nx*ne+(jj-1)*nx+1:(ii-1)*n*nx*ne+(kk-1)*nx*ne+jj*nx);
            G2{ii}(:,(nx+jj-1)*(nx+ne+1)+kk) = xesoln((ii-1)*n*nx*ne+nx*nx*ne+(kk-1)*ny*ne+(jj-1)*ny+1:(ii-1)*n*nx*ne+nx*nx*ne+(kk-1)*ny*ne+jj*ny);
        end
    end
end

% Second order for e,e
[Mee,Nee] = solution_getmatrices_ee(GLOBALS,THETA,xss,yss,sss,Pss,H,G,G2);
Meefull = zeros(ns*n*ne*ne,ns*n*ne*ne);
Neefull = zeros(ns*n*ne*ne,1);
for ii = 1:ns
	iiblock = (ii-1)*n*ne*ne+1:ii*n*ne*ne;
    for jj = 1:ns
        jjblock = (jj-1)*n*ne*ne+nx*ne*ne+1:jj*n*ne*ne;
        Meefull(iiblock,iiblock) = Meefull(iiblock,iiblock) + Mee{jj,ii}(:,1:n*ne*ne);
        Meefull(iiblock,jjblock) = Meefull(iiblock,jjblock) + Mee{jj,ii}(:,n*ne*ne+1:(n+ny)*ne*ne);
        Neefull(iiblock) = Neefull(iiblock) + Nee{jj,ii};    
    end
end
eesoln = -Meefull\Neefull;
for ii = 1:ns
    for jj = 1:ne
        for kk = 1:ne
            H2{ii}(:,(nx+jj-1)*(nx+ne+1)+nx+kk) = eesoln((ii-1)*n*ne*ne+(jj-1)*ne*nx+(kk-1)*nx+1:(ii-1)*n*ne*ne+(jj-1)*ne*nx+kk*nx);
            G2{ii}(:,(nx+jj-1)*(nx+ne+1)+nx+kk) = eesoln((ii-1)*n*ne*ne+nx*ne*ne+(jj-1)*ne*ny+(kk-1)*ny+1:(ii-1)*n*ne*ne+nx*ne*ne+(jj-1)*ne*ny+kk*ny);
        end
    end
end

% Second order for e,c
[Mec,Nec] = solution_getmatrices_ec(GLOBALS,THETA,xss,yss,sss,Pss,H,G,G2);
Mecfull = zeros(ns*n*ne,ns*n*ne);
Necfull = zeros(ns*n*ne,1);
for ii = 1:ns
	iiblock = (ii-1)*n*ne+1:ii*n*ne;
    for jj = 1:ns
        jjblock = (jj-1)*n*ne+nx*ne+1:jj*n*ne;
        Mecfull(iiblock,iiblock) = Mecfull(iiblock,iiblock) + Mec{jj,ii}(:,1:n*ne);
        Mecfull(iiblock,jjblock) = Mecfull(iiblock,jjblock) + Mec{jj,ii}(:,n*ne+1:(n+ny)*ne);
        Necfull(iiblock) = Necfull(iiblock) + Nec{jj,ii};    
    end
end
ecsoln = -Mecfull\Necfull;
for ii = 1:ns
    for jj = 1:ne
        H2{ii}(:,(nx+jj-1)*(nx+ne+1)+nx+ne+1) = ecsoln((ii-1)*n*ne+(jj-1)*nx+1:(ii-1)*n*ne+jj*nx);
        G2{ii}(:,(nx+jj-1)*(nx+ne+1)+nx+ne+1) = ecsoln((ii-1)*n*ne+nx*ne+(jj-1)*ny+1:(ii-1)*n*ne+nx*ne+jj*ny);
    end
end


% Second order for c,x (use Young's theorem)
for ii = 1:ns
    for kk = 1:nx
        H2{ii}(:,(nx+ne)*(nx+ne+1)+kk) = xcsoln((ii-1)*n*nx+(kk-1)*nx+1:(ii-1)*n*nx+kk*nx);
        G2{ii}(:,(nx+ne)*(nx+ne+1)+kk) = xcsoln((ii-1)*n*nx+nx*nx+(kk-1)*ny+1:(ii-1)*n*nx+nx*nx+kk*ny);
    end
end    
    
% Second order for c,e (use Young's theorem)
for ii = 1:ns
    for kk = 1:ne
        H2{ii}(:,(nx+ne)*(nx+ne+1)+nx+kk) = ecsoln((ii-1)*n*ne+(kk-1)*nx+1:(ii-1)*n*ne+kk*nx);
        G2{ii}(:,(nx+ne)*(nx+ne+1)+nx+kk) = ecsoln((ii-1)*n*ne+ne*nx+(kk-1)*ny+1:(ii-1)*n*ne+ne*nx+kk*ny);
    end
end    


% Second order for cc
[Mcc,Ncc] = solution_getmatrices_cc(GLOBALS,THETA,xss,yss,sss,Pss,H,G,G2);
Mccfull = zeros(ns*n,ns*n);
Nccfull = zeros(ns*n,1);
for ii = 1:ns
	iiblock = (ii-1)*n+1:ii*n;
    for jj = 1:ns
        jjblock = (jj-1)*n+nx+1:jj*n;
        Mccfull(iiblock,iiblock) = Mccfull(iiblock,iiblock) + Mcc{jj,ii}(:,1:n);
        Mccfull(iiblock,jjblock) = Mccfull(iiblock,jjblock) + Mcc{jj,ii}(:,n+1:n+ny);
        Nccfull(iiblock) = Nccfull(iiblock) + Ncc{jj,ii};    
    end
end
ccsoln = -Mccfull\Nccfull;
for ii = 1:ns
    H2{ii}(:,(nx+ne+1)^2) = ccsoln((ii-1)*n+1:(ii-1)*n+nx);
    G2{ii}(:,(nx+ne+1)^2) = ccsoln((ii-1)*n+nx+1:ii*n);
end


for ii = 1:ns
    if max(max(imag(H{ii})))  < 1e-10;  H{ii}  = real(H{ii}); end
    if max(max(imag(G{ii})))  < 1e-10;  G{ii}  = real(G{ii}); end
    if max(max(imag(H2{ii}))) < 1e-10;  H2{ii} = real(H2{ii}); end
    if max(max(imag(G2{ii}))) < 1e-10;  G2{ii} = real(G2{ii}); end
end

if printtime == 1
    disp(['Model Solution Time  = ' num2str(toc(modelsoln))]);
end
